;;;
;;; scmail - a mail filter written in Scheme
;;;
;;; Copyright (C) 2002-2004 Satoru Takabayashi <satoru@namazu.org> 
;;;     All rights reserved.
;;;     This is free software with ABSOLUTELY NO WARRANTY.
;;;
;;; Permission to use, copy, modify, distribute this software and
;;; accompanying documentation for any purpose is hereby granted,
;;; provided that existing copyright notices are retained in all
;;; copies and that this notice is included verbatim in all
;;; distributions.
;;; This software is provided as is, without express or implied
;;; warranty.  In no circumstances the author(s) shall be liable
;;; for any damages arising out of the use of this software.
;;;

(define-module scmail
  (use srfi-1)
  (use srfi-13)
  (use srfi-19)
  (use gauche.regexp)
  (use gauche.parseopt)
  (use gauche.parameter)
  (use file.util)
  (use scmail.config)
  (use scmail.util)
  (use scmail.mail)
  (use scmail.mailbox)
  (use scmail.mh)
  (use scmail.maildir)

  (export scmail-filter scmail-main
          scmail-command-log scmail-error-log scmail-log

	  add-filter-rule! add-bayesian-filter-rule! valid-rule?
          copy refile forward redirect remove
	  set-match-data-replace-rule! 

          ;; for backward compatibility
	  command-copy command-refile command-forward
	  ))

(select-module scmail)

(autoload scmail.bayesian-filter mail-is-spam?)

(define (scmail-log-to-file prefix fmt . args)
  (with-error-handler
   (lambda (e) '())
   (lambda ()
     (call-with-output-file (scmail-config-get-path 'log-file)
       (lambda (port)
         (apply format port (string-append prefix fmt) args))
       :if-exists :append))))

(define (scmail-log fmt . args)
  (let* ((tz-offset (date-zone-offset (time-utc->date (current-time))))
         (fmt (if (safe-rxmatch #/\n$/ fmt)
                     fmt
                     (string-append fmt "\n")))
         (prefix (format "~a~a~2,'0d:~2,'0d "
                         (sys-strftime "%Y-%m-%dT%H:%M:%S" 
                                       (sys-localtime (sys-time)))
                         (if (< tz-offset 0) "-" "+")
                         (round (/ tz-offset 3600))
                         (round (/ (modulo tz-offset 3600) 60)))))
    (apply format #t fmt args)
    (apply scmail-log-to-file prefix fmt args)))

(define (scmail-command-log command src . dest)
  (let* ((dest (get-optional dest #f))
         (message (if dest
                      (format "~a: ~a -> ~a" command src dest)
                      (format "~a: ~a" command src)
                      )))
    (scmail-log "~a" message)))


(define (scmail-error-log fmt . args)
  (apply scmail-log (string-append "error: " fmt) args))


(define (cut-mailbox-part file mailbox)
  (let ((m (safe-rxmatch #`"^,|mailbox|/+" file)))
    (if m 
	(rxmatch-after m)
	file)))

;; basic mail operation
(define-method copy ((mail <mail>) folder)
  (let1 folder (replace-param folder)
        (let* ((file (scmail-mail-query mail 'file))
               (mailbox (ref (scmail-config) 'mailbox))
               (src (cut-mailbox-part file mailbox)))
          (let* ((out-file (scmail-mail-copy mail folder))
                 (dest (cut-mailbox-part out-file mailbox)))
            (scmail-command-log 'copy src dest)))
        :next))

;; basic mail operation
(define-method refile ((mail <mail>) folder)
  (define (refile-internal mail folder)
    (if (scmail-mail-from-stdin? mail)
        (let1 new-name (scmail-mail-copy mail folder)
              (scmail-mail-remove mail)
              new-name)
        (with-error-handler 
         (lambda (e)
           (scmail-error-log "refile: ~a" (ref e 'message))
           (scmail-mail-copy mail folder)
           (scmail-mail-remove mail))
         (lambda ()
           (scmail-mail-move mail folder)
           ))))
                        
  (let1 folder (replace-param folder)
        (let* ((file (scmail-mail-query mail 'file))
               (mailbox (ref (scmail-config) 'mailbox))
               (src (cut-mailbox-part file mailbox)))
          (let* ((out-file (refile-internal mail folder))
                 (dest (cut-mailbox-part out-file mailbox)))
            (scmail-command-log 'refile src dest)))
        :last))

(define-method forward-internal ((mail <mail>) address command)
  (let1 folder (replace-param address)
        (let* ((file (scmail-mail-query mail 'file))
               (mailbox (ref (scmail-config) 'mailbox))
               (src (cut-mailbox-part file mailbox)))
          (scmail-mail-forward mail (ref (scmail-config) 'smtp-host) address)
          (if (eq? command 'redirect) (scmail-mail-remove mail))
          (scmail-command-log command src address))))

;; basic mail operation
(define-method forward ((mail <mail>) address)
  (forward-internal mail address 'forward)
  :next)

;; basic mail operation
(define-method redirect ((mail <mail>) address)
  (forward-internal mail address 'redirect)
  :last)

;; basic mail operation
(define-method remove ((mail <mail>))
  (let* ((file (scmail-mail-query mail 'file))
         (mailbox (ref (scmail-config) 'mailbox))
         (src (cut-mailbox-part file mailbox)))
    (scmail-mail-remove mail)
    (scmail-command-log 'remove src)
    :last))

;; backward compatibility
(define (command-copy config mail folder) (copy mail folder))
(define (command-refile config mail folder) (refile mail folder))
(define (command-forward config mail folder) (forward mail folder))

;; (set-match-data-replace-rule! #/\./ "-")
(define match-data-replace-rule (make-parameter #f))
(define (set-match-data-replace-rule! rule)
  (match-data-replace-rule rule))

(define last-match-data (make-parameter #f))

(define (replace-param param)
  (if (last-match-data)
      (regexp-replace-all #/\\([\d+])/ param
                          (lambda (m)
                            (let ((str (rxmatch-substring 
                                        (last-match-data) 
                                        (string->number 
                                         (rxmatch-substring m 1)))))
                              (if (match-data-replace-rule)
                                  (let* ((pattern 
                                          (car (match-data-replace-rule)))
                                         (replacement 
                                          (cadr (match-data-replace-rule))))
                                    (regexp-replace-all 
                                     pattern str replacement))
                                  str))))
      param))

(define (process-command mail command param)
  (cond ((eq? command 'refile)
         (refile mail param))
        ((eq? command 'copy)
         (copy mail param))
        ((eq? command 'forward)
         (forward mail param))
        ((eq? command 'redirect)
         (redirect mail param))
        ((eq? command 'remove)
         (remove mail))))

(define (match-rule? mail field-name pattern)
  (last-match-data #f)
  (if (regexp? pattern)
      (find (lambda (field-value)
              (last-match-data (safe-rxmatch pattern field-value))
              (last-match-data))
            (scmail-mail-query mail field-name :multi-field))
      (find (lambda (field-value)
              (string-contains field-value pattern))
            (scmail-mail-query mail field-name :multi-field))))

(define (process-rule mail field-name patterns command param)
  (if (null? patterns)
      :next
      (let* ((pattern (car patterns))
	     (status (if (match-rule? mail field-name pattern)
                         (process-command mail command param)
                         :next)))
	(if (eq? status :next)
	    (process-rule mail field-name 
			  (cdr patterns) command param)
	    :last))))

(define (apply-rule mail rule)
  (let ((field-name (car rule))
	(field-rules (cdr rule)))
    (let loop ((field-rules field-rules))
      (if (null? field-rules)
	  :next
	  (let* ((pattern-and-command (car field-rules))
		 (patterns 
		  (if (list? (first pattern-and-command))
		      (first pattern-and-command)
		      (list (first pattern-and-command))))
		 (command (if (list? (second pattern-and-command))
			      (first (second pattern-and-command))
			      'refile))
		 (param (if (list? (second pattern-and-command))
                            (if (= (length (second pattern-and-command)) 1)
                                #f  ;; for "remove"
                                (second (second pattern-and-command)))
			    (second pattern-and-command))))
	    (let ((status (process-rule mail field-name
					patterns command param)))
	      (if (eq? status :last)
                  :last
                  (loop (cdr field-rules)))))))))

(define-method object-apply ((mail <mail>) field-name pattern)
  (match-rule? mail field-name pattern))

;; Filter a mail
(define (scmail-filter mail)
  (define (scmail-filter-iter mail filter-rules)
    (unless (null? filter-rules)
            (let* ((rule (car filter-rules))
                   (status 
                    (with-error-handler
                     (lambda (e) 
                       (scmail-error-log "in a rule: ~a" (ref e 'message))
                       :next)
                     (lambda ()
                       (cond ((procedure? rule)
                              (if (= (arity rule) 2) ; backward compatibility
                                  (rule (scmail-config) mail) 
                                  (rule mail)))
                             (else
                              (apply-rule mail rule)))))))
              (unless (eq? status :last)
                      (scmail-filter-iter mail (cdr filter-rules))))))
  (unless (is-a? mail <mail>)
          (scmail-eformat "<mail> required but got ~a" (class-of mail)))
  (scmail-filter-iter mail (filter-rules)))


(define filter-rules (make-parameter '()))

(define (valid-rule? rule)
  (define (valid-pattern-part? pattern-part)
    (or (string? pattern-part)
        (regexp? pattern-part)
        (and (list? pattern-part)
             (every (lambda (pattern)
                      (or (string? pattern)
                          (regexp? pattern)))
                    pattern-part))))

  (define (valid-destination-part? destination-part)
    (or (string? destination-part)
        (and (list? destination-part)
             (or (and (= (length destination-part) 1)
                      (symbol? (first destination-part)))
                 (and (= (length destination-part) 2)
                      (symbol? (first destination-part))
                      (string? (second destination-part)))))))

  (define (set-of-rules? field-rules)
    (if (null? field-rules)
        #t
        (let1 field-rule (car field-rules)
              (if (and (list? field-rule)
                       (= (length field-rule) 2)
                       (valid-pattern-part? (first field-rule))
                       (valid-destination-part? (second field-rule)))
                  (set-of-rules? (cdr field-rules))
                  #f))))

  (if (or (and (procedure? rule)
               (or (= (arity rule) 1) (= (arity rule) 2)))
          (and (list? rule)
               (symbol? (car rule))
               (list? (cdr rule))
               (set-of-rules? (cdr rule))))
      #t
      #f))

(define (add-filter-rule! . rules)
  (for-each (lambda (rule)
              (if (valid-rule? rule)
                  (filter-rules (append (filter-rules) (list rule)))
                  (scmail-error-log "invalid rule: ~s" rule)
                  ))
            rules))

(define (bayesian-filter mail)
  (and (mail-is-spam? mail)
       (refile mail (ref (scmail-config) 'spam))))

(define (add-bayesian-filter-rule!)
  (add-filter-rule! bayesian-filter))

(define (read-filter-rule file)
  (with-error-handler 
   (lambda (e)
     (scmail-error-log (ref e 'message))
     '())
   (lambda ()
     (call-with-input-file file
       (lambda (port) (load-from-port port)))
     (filter-rules))))

(define (show-help program-name)
  (format (current-output-port) "Usage: ~a\n" (sys-basename program-name))
  (print "  -c, --config=FILE    use FILE as a config file")
  (print "  -r, --rule=FILE      use FILE as a rule file")
  (print "  -f, --folder=FOLDER  refile mails in FOLDER")
  (print "  -d, --scmail-dir=DIR set scmail's directory to DIR")
  (print "  -n, --dry-run        don't actually run; just print them")
  (print "                       (disables copy/refile/forward/redirect/remove)")
  (print "  -v, --verbose        work noisily (diagnostic output)")
  (print "  -q, --quiet          suppress all normal output")
  (print "  -h, --help           display this help and exit")
  (exit 0))


(define (scmail-main args main-process rule quiet-mode)
  (scmail-set-program-name! (car args))
  (scmail-check-gauche-version)
  (let* ((config-file     (scmail-config-default-file))
	 (rule-file       (scmail-config-get-path rule))
	 (target-folder #f)
         (verbose-mode #f)
         (dry-run-mode #f))
    (parse-options (cdr args)
		   (("h|help" ()
                     (show-help (car args)))
                    ("c|config=s" (file)
		     (set! config-file file))
                    ("d|scmail-dir=s" (dir)
                     (scmail-config-set-directory! dir))
                    ("n|dry-run" ()
                     (set! create-directory* values)
                     (set! scmail-log-to-file values)
		     (set! dry-run-mode #t))
		    ("r|rule=s" (file)
		     (set! rule-file file))
		    ("v|verbose" ()
		     (set! verbose-mode #t))
		    ("q|quiet" ()
		     (set! quiet-mode #t))
		    ("f|folder=s" (folder)
		     (set! target-folder folder))))

    (if verbose-mode (scmail-config-set-verbose-mode!))

    (with-output-to-port (if quiet-mode 
                             (open-output-file "/dev/null")
                             (standard-output-port))
        (lambda ()
          (scmail-config-read config-file)
          (sys-umask (ref (scmail-config) 'umask))
          (scmail-config-make-directory)
          (read-filter-rule rule-file)
          (main-process (make-scmail-mailbox 
                         (ref (scmail-config) 'mailbox-type)
                         (ref (scmail-config) 'mailbox))
                        (or target-folder (ref (scmail-config) 'inbox))
                        dry-run-mode
                        ))
        ))
  )

(provide "scmail")
