// Copyright 2025 Google LLC
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package mldsa

import (
	"bytes"
	"crypto/rand"
	"encoding/hex"
	"testing"
)

// TODO: b/401208985 - Add Wycheproof style tests.

// Number of times to run sign-verify roundtrip tests.
const numSignVerifyTests = 100

func TestDerivedParameters(t *testing.T) {
	pars := []struct {
		name       string
		par        *params
		expEtaBits int
		expW1Bits  int
	}{
		{"MLDSA44", MLDSA44, 3, 6},
		{"MLDSA65", MLDSA65, 4, 4},
		{"MLDSA87", MLDSA87, 3, 4},
	}
	for _, par := range pars {
		if par.expEtaBits != par.par.etaBits {
			t.Errorf("etaBits = %v, want %v", par.par.etaBits, par.expEtaBits)
		}
		if par.expW1Bits != par.par.w1Bits {
			t.Errorf("w1Bits = %v, want %v", par.par.w1Bits, par.expW1Bits)
		}
	}
}

func TestKeyGenFromSeed(t *testing.T) {
	pars := []struct {
		name  string
		par   *params
		seed  string
		expPk string
		expSk string
	}{
		{
			"MLDSA44",
			MLDSA44,
			"000102030405060708090A0B0C0D0E0F000102030405060708090A0B0C0D0E01",
			"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",
			"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",
		},
		{
			"MLDSA65",
			MLDSA65,
			"000102030405060708090A0B0C0D0E0F000102030405060708090A0B0C0D0E02",
			"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",
			"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",
		},
		{
			"MLDSA87",
			MLDSA87,
			"000102030405060708090A0B0C0D0E0F000102030405060708090A0B0C0D0E03",
			"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",
			"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",
		},
	}
	for _, par := range pars {
		seedBytes, _ := hex.DecodeString(par.seed)
		var seed [32]byte
		copy(seed[:], seedBytes[:])
		pk, sk := par.par.KeyGenFromSeed(seed)
		pkString := hex.EncodeToString(pk.Encode())
		skString := hex.EncodeToString(sk.Encode())
		if par.expPk != pkString {
			t.Errorf("KeyGenFromSeed(%v).pk = %v, want", pkString, par.expPk)
		}
		if par.expSk != skString {
			t.Errorf("KeyGenFromSeed(%v).sk = %v, want", skString, par.expSk)
		}
	}
}

func TestSignDeterministic(t *testing.T) {
	pars := []struct {
		name string
		par  *params
		sk   string
		msg  string
		ctx  string
		exp  string
	}{
		{
			"MLDSA44",
			MLDSA44,
			"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",
			"48656c6c6f20776f726c64",
			"436f6e74657874",
			"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",
		},
		{
			"MLDSA65",
			MLDSA65,
			"f5408337d0fee65c28851226a5fa81b58464632c78e2a9bef70d330f2e3a5f743fe2130a9185402b9ceee616adcf0aeb1b0adc5b7bd65b8469d9f73abdd491c7659d92d3c6412cab96a2de13921e6e28264326efab5964afee6490ef637e546aa6b61aba997daac4623b0ff33b4bbd1a11b2942500c737263b5a7b599c454d8442073855220447142672283106762324611334160331766716756876656306340247416643645788214401245418430332335014484266482863377264642414206877226544283410340543730247627225170453086122730061663720413006672700563545553308822632620516184200008121626450075764267612441006130131054105366434124552571655324024583437672387040523848872722438820242507702366302474748842265201861837063577540747600354316761325342724414244818807083666356763458818178730038860216678417556380528118134053806822634757706213780383068245037164828641644088712486666263228563278454077257226514241838623545046705775223416234252343326255430503041432773656045076111030080425435056385571604503837242375546781881637734567572558705071178084805270422810171125366178026263016101361657832287864742571181540760851116286862663367644442262567158480021516651834322735675322048815731072442030787610367014103700472855221118750731347147044142277174111214348353566824008064045627684488722675430832058628153456611117540326632627423710448122776364844470152004860248280007708325718482525552264175551177652286873852661125522335557583772044467545176204662306421267463234564162341677110818645117237332101582140278868362167651433830007175820104352222533408675507171265571436206825831530330505138722014332544414101453361865663285823831165654236338452842635115828271801121281582618708170531302365851580567783463004004872080286853401722161683240507540174321162726623307704357168735441623362805550771378557544025577664653883727145477726652161328654636765354441580462755002525303617004457666056523833318532171037664380506224158805461788342412558636736557807030038168058241037440287466117173258168248137753751448321557734277603877752226522005384287417756673375054253360826560561501724117752720403752166506773562110186547571130831846605687668266432088723466508134167088157733748882573483206443607642671007311272002682616013071771238832745526182648607836043640787500384324825714344453303465740312784685175588706426626253688784802024002168480055006521746017031355351420356313376038357500434586744155331653510137233135501164040340443180537652161358720413477471755650162313051181844240248360407773704517351058764775404747121083306411352302550060311617664848461285205032215087833220558462453850275140274158215805155381750850757850271010637641250528012682211173613444785706268032670127024707747567314772632848841668565545456641117066848041360116507056675046470163210801085300085527862576471877326436205146166232414075335513375673166307774724018723351488672332685528233522486007870307457262016571753124063541231514175538351737543321726247362412023101043212535074048080016006886215511021520172701460518406542024738500552450424583471822538241124124282248288348872240758184343246825136527273210452765138144072540628287154522674671801747806751802628540570743817583756538088277258068473227163621044527db3e8a7b37950eaf9af3770bd6b8f8c9572b54256463bebfcfa8920624daa12fccd1c228c4668f688976a5094a00b6e30694ab2e017cdc2d4315a381bbe84ee58340cd6bc6fb225ddc2bf8dc117ce2be59993e5a521f5296126f290d10097bb2b9f171ff3f5de8a5047e76a15594c3caddc7c0caa852b8aff91f435d3f039d3f27963c357691e9c9ee543f32a77691f8b8ad98d711f8191f5c9891f593d8c754b07621a42cd25a2e95c32156a356f7bf71d6a3d1be00c7364741f2eac5d0c5fb3ef93c6339b559c9b2c4f518927fbe1b87d49546d0c9ba801c224a1cca84916aac512fbaef6841f725c72b927b55d920bb9b8df4e89b9b09da7adc24fa3f9834a8a0380bd42f07801e4afac2bc1e852dfe41b0903990289190ddcdcc214a680e0247938a4b54b704bb6f0affe7ce2171ab7e05f8cb96abc352e44e9b9f60f6ceba0ff95703209197657d8228e93dae9a9f478a30700a6cdddb4ad484f963afacb39a3186e94e097fa25ab149c4c438458b24369f62f87159396f37de36fbcdae2acc7b996a90f66c6de64ad905ba6d4508d3c453e7d43ecb7c8692244f3de88c975046189a438f61a1a84e8d0eb11f93dae6c80dee63c2a3ece5b794483d60493a84dfd4d267eaa205ad0b9491e656e267eec42f5d3094982452ca10b420552d684242bb72d651d47e96da49077d73dec8fcbaeda2ace53156bb4d4392e21fe094a2a32d3eb3d4163ab50a8528800464c05cf2db162da56dc13e709844dda7cbae47ef8364c8c3d5fb471f9407bffafef778486c304088801624aa2318433459b4794f4c31ec2cac49d56504602db94a5b882a92b06ba9ea1ac40523fbf4ec1dd473c076d3e397941c821895beeb4b5fd5eef9246ed1ac724a2aeab43c56a6584bffcf4aed0c6629d5a59b719e5d677539c3dd620d861ee7f9c0a5f116a17e4bb3b16a8017e668612c1038b576e39dedaf7d848d02024c0065f1b896ad6594af4477c73544ef51f3e612c80f11150b1f394b9fd899fa6d927535efaafb745133a45442241579e57701399ba23b0ba62a2ef13b5eead46f02d2c5a91fc031e458be09a8b5d53d9843278590bf556e2258b371f14f07cedc213302ecc895e15be2eb0e771e4b1d6d10fffd2ab945dacc52287e8667ecefaf3c959dc5bc18d0d953ce392acaa67e3f457630cf266c7665253c3c04cef5a73aedee0cfd06efc300887d4694dbbb717f2813737d0fae91e434922ea831ae41349e44ed0fe9101309dee68c1337f050e9e16113e8ba47877202b97113b3b876f4333ee620cfe37b69e597162ad40d78f5a7be33ebdb38b9eb84dea68e01de157c7e74f4522836b64d146f719df851b60fe69cea4667daa057bc47a47248fc3073a904239c0f8757cd8aa52e5d837fbc6490a0ffd9a3dfbeed09c24c61e48df0695d8f37f5670aca01a3fbb2a5fd3bd57cf38c453f80f9de428aa0d860b13231c5b2e435905bae792ee21e6104535204f9ab1f90d8c3123e57e2d01b0091ed089b9083438510579190478b86980370bb0170992673394509aea3a64d241b0d5388614282c4172796b00bcb034769bb492c0fdbc88c1a30a040d637a7b9c496f620114546133fed394ab79413268027d5418bd61e6ce3ff18ab83f76ea0a2e5694a48c01931cbd1c4bb02a3515ba3fcedad86e74b2848902089986bca91694dfa712b97c4f2878f287590aca47afb18d1ee6f502b90749f8ca1d796309c5bc96f5899f70fe0b4b5061668e62891b5285546f0dd2176649e5cba2b206953c7b66c57fe2e67c1c6e45682322d82c569c543840722756b7b4c92aa059e59983fcc327833931556346b83d7519b3a6b93691e74af6d26fb61bac9cce2f880d1f194d084171809bef926e99218f664bf18f58585985534281b6008d86f48f8c4044a4e4c89f4f08cee869c6a09b96df42d6435433d247e9e1150ac5b09da7f66d4f2eb0fc7f8f22bf11a0b03c9958f0562030d2c93b1c4a554e86b00f7c2e464aa54fe018bed4578f3ca0ee8644c3537b86da324703e7779fa897fdd4b867f7620e7fad29957f28067524ea180380428995ea333a3f291fdc67f0364cac8683e199a9a85c7caa3cc29db52f84aa0d0d07d5677d74816873e0af727a8555149d41d40a1eeb5944328a23b1be9e55ea81bfe461c86bb1fe3995e9f8dc83a77b06257c6356089bba2bd44e2a7acacf6de34262d22b0f899d0abd705637ad5ed1de2691e498851db106f1bcdada3a2f96908d3774f661fe2e8ec3ec598b936f4dc4d04b80efa4f57f3c55d0ff990b313643a8126aaa14d898e338f5a261ae5bb17b854cb1a87ad626d3b4ee29c567baaf409ad53891f707c148f49ec16f9882db396d1f140ba9e8b0c9a0cddde6592e5e82de0b18dea363e8605ca364aa68ebdc5f3220b810125fc787d339c6a5a68c92cbc64b5d265ce7a03a06a2409a8a18f000691cfbc5b205bcc1926efb589b82a7404d1bedf09e88ad286f7711a6d67ece4bf76fd86556e840d73f353d31b4d1c3dad5e83f72d540c601983cde7171e67b805bdabe3cd6788e30938b0398c1c8cdd5f8f1028f123d2d612611ce15f7637d2d751e59df171a4014efa4e8420bfcdb75b77c63ad1dfde967d4cd44792d1868d6df3c5bb5f1f3617861e24ee6fb58c33e6b82a57b29422dc0c5aafcf0ccf44557ede0e9f7d32be95f4a214524a30d37b85f1783c3e77b7d52e8648afbb9a03435c59d6afc2ed3b6ee434620c0b368ca6337cc7228916c5857d1bef27502014a0798ab099a7d7d9f053175b9e94d4e3736794034bcc92a842edd555dc74e69906a8021430655b3907ff0d5af127c847902c783e5e51bb45a3e4da6bca417774417fe2cd5d8448c5422eb55ebff2a44a378d528ab06419e6ee4ff6608eca6b44b8c71f55faa714bff9727dd3163b090330713d90813375c88b8681d491527f89b87f9433cce8f6dcd83fbf4f00215b3caef344b38ab4bd9bc5e3dcc3624fb60ce34acd7340b14d9ab48df15b1cc839265b791a417f90cd463df719fd69869aafec64df9bf010498e8d6397e1cb696fd68b38ca9ab1faf67b7123ad01781ffce12a3dc2b703cfa73dce16de7c3a09eead7ed6a644f96ae33a4bf179122906c969e99d9d99dc3472de3c48e80be4e9685602b23400bc292ad6f2fd34a2afb87660a010fec11d3ee82c164fb267957b6651d9150cfba16650952f8f8863709bf5644b0e4738c45199824511fae580548c7f655f0bf0bf3102002d80c2b2c3adfdf358f2a38346032120519b31dd906d3e26518b867d1dcb91caa5567976cada346044dc8e7274c0bdc0597556ad059f711868b83b8da5920304d08cdcfd3d5d427287f5b637fb7e2c14253452915fd6c6614f93c3900c6b3469d7bd87e2e5f15629316a7e8d4dc12ac80a89bb8c6ae7238144f82792bd04c2e741beb130c2a5a9caeb09e8f02b1480ad42486d0d8e5229072dc139115493c2c747920ad76391dddfd66feac8744d5662014bce98728b8d",
			"48656c6c6f20776f726c64",
			"436f6e74657874",
			"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",
		},
		{
			"MLDSA87",
			MLDSA87,
			"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",
			"48656c6c6f20776f726c64",
			"436f6e74657874",
			"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",
		},
	}
	for _, par := range pars {
		skBytes, _ := hex.DecodeString(par.sk)
		sk, err := par.par.DecodeSecretKey(skBytes)
		if err != nil {
			t.Errorf("DecodeSecretKey(%v) failed: %v", par.name, err)
		}
		msgBytes, _ := hex.DecodeString(par.msg)
		ctxBytes, _ := hex.DecodeString(par.ctx)
		got, err := sk.SignDeterministic(msgBytes, ctxBytes)
		if err != nil {
			t.Errorf("SignDeterministic(%v) failed: %v", par.name, err)
		}
		expBytes, _ := hex.DecodeString(par.exp)
		if !bytes.Equal(expBytes, got) {
			t.Errorf("SignDeterministic(%v) = %v, want %v", par.name, par.exp, hex.EncodeToString(got))
		}
	}
}

func TestVerify(t *testing.T) {
	pars := []struct {
		name string
		par  *params
		pk   string
		msg  string
		ctx  string
		sig  string
	}{
		{"MLDSA44", MLDSA44,
			"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",
			"48656c6c6f20776f726c64",
			"436f6e74657874",
			"e11d24772c24efc107ae3abb0149817436f11684d3548748cba19fc0b373ddcb7c8f68f00407d964570c155a9a34823d5b33345a2bb4dfc43d2e178331bc6573f39d634239230cfc160bf03f41d176854dfee5be915ed6c3f4112fff50d8effcc457708261e715fdf0676831989a15cbd16b92fc97bec06c75919c114c167d2bfae8d7dfa384068c0d96a8e6039e755f9b90cb57b4b0e678854a88a8fada69b91bbbea873f81a7489c0e3612774e8a00370b9b9650331bd2184b9037ce340d82b39436dab990f0c176b90421e71fd182bc07ed70e54587bf2b92c038e8794aded666a6c9cdb29d8747c223967c5a283d3be2946584202a021c5264e04587b3c60bb5ec7a73e2d4d7caf4619e388d1beff4ec4bf7d104fee34765ab6a51108660f052a05d16aa46efc49d46ff42d65bbc6521d8a18c8cbe104de453367bae5c72b43854def8222480746003fc8ec4efa2d122965ef9e0e5b3d68c9069af54ef4511036a079d9bb67a43eabec138d37eeaa918bf14815159b0216352a354110d5c835ea9631075317ba617085f2d86215c09c288a584add2809bcc7f50f9071fee5ea2fc08020f2a106fad222155155018f67162855ce624328724b659c645cc30c6382c6fdf48e1c9e8499bf6f8ccd63f06113e3262efd0800d2619d59cd8966d847c2de3854634f3b5e83f84e66cac84e1013b93fe3869f270380ccf8c26591a2635cfa048d1955516560c95ce0c39b0cd7c12c3234b13939386adcf557118f21811c3595151919da2bce155f9c6300703a7209fcd893305486df90a828bc551f23878b72f04fe471ed75982175b74ce135fbdf0c786acdefb09829afdaf7eab308cd8c181345e8f713afd5b433a6be59a4e70b421c216a02a16bf0e927630992211d48d71ac0aec3d0626d84456303c3f35c132571eeafa0106cc7ff333e0d2dcd9352b3cdf36a8fec2a750e5c8ebfeed52a94e5f41c1d295ddc01de6ddbf9df9970460f33fb362b0b94fac9b496459c6ca989e90d53ec8944d1518d7fcc21f1adca0bac93df266820dfbe9c7cbce4b762340ef8ea6464d26c5fd4f2b67b9776548b567d7426511aa9c2fdd19d85206130ab6cf6d7f5115dcb7f53b628b99ed8fa1bd6055764f950deeabae276b419370c4700cd37ca2a34b387d644d4e0ef6a380a5e2d2f32376b4b8752bfc3003c2b67111105b775fd21c3e5ae678f79975097e6c63e759eae6b14d60c9778b4bc31aaa4c9f4fa4911688dc390047aa11f9a998baa652eb9be561cb4039bd9801fd62eedb6f568ff4189dffa4c9a7bc11d9faf26499285098043fe699b565545a930d9ce8f5247eea4c5f6df27f3e050b8d01eee5dd1058efe65190eebeaa0742515d9f8f36bd29e6d84e56d9e41c1a551d3ce6ad7e8967872abd60488d4172c56006eb2db95cb25743287a1d73fb3a36ca4d7f7dce22fd2baf10ad47aeacf82b37dafad7c06a6795be40bd6abfc8f998219f2a0e58531c8ccd1bf3ce66b960741a2da9d36971bad67ee4d75e660e0805e889eab0f0be62b38439476ec289e77176341461b474f66f44120f784de5490529a1f6f013eac2dfbdea11275733f1b1723357740a903085e09e8d61a2e2c84f26ddf95fe630a398329e48cd58cbf358b98b839c7f17893b6e913ee286c976bea3a0bbc58177ce0a35a28c5bb4ac6d9d5ffdb9dc626555a55bea17386237d8ccf2ef60a31393b1f49a37329598f706eeeca9c2d0b02ef13dfa6bb9f1e84517aa51d7d7e85ffbdacf23892962d231f67c142df49d6236630bdb50dad047bc84fec4f517758c3f54c77f5f25fe78a12db9e4dd766198d6014b35cdbab0257cc50c7f9dfa5ac0a88c7d107c8f6bb50dee4d7a3e35cc54fb12572d901f02f4e8bf15cb6fef1910fcd5d54530dbca4046bd9ba3039c4ff97bcbfb6d00a16c1f902a25005c30d3d0d96a9d7116b15f81699614afe0aa448973b6da55c18f20395a15d2ac53c5725e45711f9b3050ca8f409d4776b568afa8d6657668e7d6d3553d23bdbe09cd1957fc5c76fb733b237e60073dfff5d64ad3f03d3116fe1db0ee27c36b9671b0efa079cb0ae0558023ac6a0aa36f1f2d887805658131398f78b4c2fb2e0bfc4a37e444015879f0db10abd5b56d5993a3ccc0798651c0b85b658285cd00e898be4406a431e29d861379c26ed26cee7f23c05fba0519fa6d0336120dffd6d441d7de14233ff6c345425b852e1cbef6ac4d442e6f121975b912b9e60538b5efe74c3df3861671b54d96d1d512725fe63b511c4d90261577f8a992746cfe6a4e1426a3d9fcbdb3098a626681ed5c41c3158667708c321a515a978c47c337b1d9cdf6be83fae368d57843baaea2b8b7a94398a8fcdb3b3e39c55a8feceae53f4b2b8967f5a7f671d7cff584596682ed7436979ee9e8610bdcdd0c065b39e22b3fefdb8ebbe7ea59ddb2058980f8c186ec95428a8cea2c41376312a073543283f2c8a970b11f1f31dc531748292cf198c63b2f21996f2bf769d397083f5f7c2da8952b38a199a2fa2698e156cc5550f123d99d4f65852fab97e184f0f615ac419af60c236f4e1c3c209b4eda22ec47c963d6b5318031cda0b1ce9dd0876b0a011d9d1a8a1233c38538581401dcb8766c4c9147d257828a0068a91e458e3a312e398c2b1affcbd7a702efdcb3f79a28d131667545f2ac3d04fefee0228f257e689a85fb92f528d901768a2dfda51f65ad31e1b781759cde2a44adf0a4b84639a8160bf863445f94a04ab7885fa247fe057c161246f1202bad84345aea9e34b77ef93fe01d090f49e1ba3e214acfea26bc04e4bb2ef2f4fa2af4751a873573ee273d8ab7f1d59aad74c8da98232e2562966b6816f01c1db37c0b5a55710011656ff76f8eb4bbba1e5875e954f1dc43bbd0d77b09cfbc57890acedf796507d31fee63305cc97209964cc7897befd20db3d6203a317bc8769b8b0081016f2180eb3b40d24ac1458d0afb8034b8babe87c91ead17f25715104be58a526409e8f5053b67e48d7de17a2f81f68a679a6d9192120eda7564c7970c88d4aa266f7063d6b24de7b402c69d9d14f8d51b3bdff45e952c45ead4e729d195f930870fda380f64085011fff63caca5e79d1dae0b2b0dad7e01c4b7b2714b20d3bb69dcee4fe9e0412420b55abba95bacbc1b1fe498474d8d3a5396968b057b8b5081ddb57eaae581da0a1b482879cdc1bda82fe83d4007375831cf06bcd334ac42c780cb91121eb4021f39f9292a6a023b1010b35d378a798601cd4a6cfebc0f45b1e7879a8f884e3d465a6680a0b8cbd5e0f210111a40464a586184859299a5e329384751b5c5c6eceef1ff1d3864879398a1b5b7cbd80000000000000000000000000000000000000000000000000000000000000000000b19242f",
		},
		{"MLDSA65", MLDSA65,
			"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",
			"48656c6c6f20776f726c64",
			"436f6e74657874",
			"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",
		},
		{"MLDSA87", MLDSA87,
			"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",
			"48656c6c6f20776f726c64",
			"436f6e74657874",
			"62217e84cf857b184906ba906b223394356d695808a3beac5c9df51dee7b21500679e5b9ca4e4abc35c150ed56356d3050f559d54c176f57139afaedf4b688d440f5db3eb9e5fc6adfdf9767a53ca1db1e74b9072bcfefae88b4d0f2c6de415ef52621f4adb28ef54a0c6e86049548a5b2152f56d4c394114bd088ffdae4fcbecc2ada3b0ed796b5c5e4e46ba101d4082137d93c9d0ad31c5e6117cf4bccdcff664b0d284c994d9ad7a523c9eb202df60c5a1213489af6a13cf4870d60825e41ee282f1012cec3a69edab1c7ab5747c357fb084c699a6f02ba11728b25de7fa9de00cefefe82a62127dbed26a94c4618ddd47e8f781c21c38d42c5c29ae7e0e4250646396bdbb9537327e1f13fde1e4e5ec2aacfbab95204f9a623a625bb4bdcf04acb1c26f36c77909468c733bc379e7c5d9aabdbcbe21ec13c955d461a6721936d7850e2b13bd72947493af86d7dc955fadf1963dff6ae298606ca365f9f89d32c405d8783cd9450a0263733d13dc71223a019fc23b0bd9f80b031961b4088f50b04eab8000bf76a550f28058e6f6b9f5464d8c83d3114223dd29803ec0e4a7af815e21d97a6613bbd21476bd383792b3d3d65eb7cb87b3e18a883ab2051e7d8856237859eb352a24525f48258374615ed9983d3a9f0a30295d6f2e43919ce321d8508796646834d16803eb2185de0374bd61f3f0875e569635380bd063e5a66145289a80ed002c0474b4dfa8e0a2d7e3659684e2a3298fed67e53dd75f3024326076d3df7bf0b8554b46e271fd2c1075fd86df402fa2069964d1748f44f0fa79931335cf2855724f50ff62d72e316a06a82d85a7a1a588291aeeaadcead38c53f1cda880bd441caae66b2588bb93c18dc5db8bad4eecd43859ab2038e5eb151061fb78c6efd9476c0fe2ce070d0b0d832d46a3dff0d52d585c71add41e3a62f4f8b0f70818064bc4b06ce7cb239186b50fa687271056786b52a1f37be5454550091591aa5cc84971c5d8f1e4b894e31e2108907d0be93a4497bdbc668efd6abc9ca9daf86413a4746288200486f2d2c607d5f09c8cd7a1bf9484cf9335fe5d06f60412563e79c41b80f3e56ea3e02391f1c4aeaca8622694cc26bb1fb6101b199f263ca81db6a75d335a814389e5df5c12fd00d510feea725a9109a1adf00912ec7f2356e55b96ee7303561f0fed9548487fe3633442df5cdb61ffc33080803a8962f413159859345871b17183e783cec8db3a2ada8ea3d55eb7ec73161073d88c654e56754d2f8d1f421048f1046672017e2229e17df61f03109e993a02102e6807f8756b0d1ee499131ea0ca8d4156004a6a0637788bca407093ac59052edfe0de9dfc53f50f545777d2bdc0bccd985013143fc464595f084e67e342271741ce38a96d4c4e3c4f6f9239aba9d2ad5c23f1fa8d7bfb60b83beb67436ad3881ff481543febebbee46f9e56b6eae7df3ce8b1f1203c38fbee440d6a51edbd10e4c8894df60312ef3261949c9d79244e47d5e98c4ec23c10eef7459d01d485df2d13403a16949df5a6603c73925eaba5fcfb49d70acbdb3742f45725f50224fd4062516b04dffe6845f78051f94366b0f78e9daa12cdbc6afe81434b7bdf9a66a81a2b68c5aa91f8dd6f02a49ede5f4e5947b33ea072790dd77159d7caa71ff67670eff0d164b1cacefbe5af46c68cc124e680f74760064df229905c670bb573a61c74dc8d948b87dc948d5b12d6327cfa3ac48270c3713f4b96ce7b9e00185ef74effca6a4994dba8a3712a986d72f8b95d00f4c17022c1ef97deea0346402723d62f8550037a604d8aafe6cd0a5063892449e94693e53b3a71dcbd9c94d8f78b1eaabdae982a5c7a23aef2b7e055851ff53206785c6270aa10c3f12df291bf61140b38603dfe75179a47736bc069abe2d115bb74585b86e2a449ad51445123eabc54070a42719e67f75c2fc411ce3d05141353cb33da779b0fa6e2aa6f9a55da5edbc85952832a39ccdd973fc7fe0fdc5c5ec4bdb68913ce8f2580507e3b5e16c4c981a14c027b8becda417cb1a9428693e9505e5a0cf5d9afc1ea715abecb49b083d05a4efd5e717bde7cbcc4eb74f3011ccc28ff31b9a74459d9fcf1d036eb78c1c85b4e56729c1d7e5135b99b89a8f238ab69b7d730b3a47da559ab45e004d844ddf7c9444241df1291e00e89e91b42d93a1bfa1168042819e28758c5d2b45c50870ae4e1d3f7a05510e72f422e684a29d71a7ba484afbc24026263078bc152446cedf142102d3395ecc6d32e98e5e6e3cab52918cbd86c11b86dd3d80c4b75035e62ad78474f5870f0fb9f421e109f0c3152891c070989d5ecc660f3c9b1bfe86694b7712c4b63cdda78380710268a72c87275641c7236e3c7c40b368f53c98d708794b469b1e19c728d2ceb836262d15f975d14ae5c822b13c3a450e5c0a4784b7b93949084597ecc4f86a06960619e92fc3f5ba1e71270f0f5f8ca6fb35aa9fad66e93c3b169af95572022900a3ac4507c005d518b7727639cbd6e315ca0473ec98cf2975df8bbf7f2ede7090801cf5931c396a5eb123a1c73471c5483fee67948dd28e5a9e6fffd612f59323a39cc3630b43c0b66a74d610236aceb442cce8c67c051cd1b56af5f299c3e90a82848eca945602289cae9ba4065e86b502147c06e8c7b76fc18f6ae33a27a1661f60cebc713802f43ebed8ea8867c69e97d4fedae2679f88579f7bd9cface5ccc1bc48eb0095659dff03f1b0babeff8a09f36e7a29a9afc69965e7b70f63c8e7a0c5da68ef6bcd074a427941045fd96c18497c124f5fcdf34b0f17b29b22110fad3d42a6399cbe6d7db7e3b0a93c3b9e71c068a9f52aa70f677147689fc79e367edd5abefe23252a4536ad1458abd91916c28fec848ebe1add87a9e5b63460a04163dc8af8051b24193b1d38ad64740d10ee2bd426e36548edb8183e8a2bf788b0a8323217ec286054647d1996f4f88772f11f4eb64b64d0a240c332de938756a442ebce2a6602c01bd382f5c532a574c5cb3401187590c83f84708b72600e87abc0e77a1093b5d0a70c8bd62067a7bb7a2ad7350c1543817c62c5bfbb6fd3b45fee0d77496ff7e3a8b5945da9990fca764849257f4b95f1e266b8fdf433fba3815c250253def569dae59a366d6cfec2b4b07c907fd48dec46be900505b1cf58f0c43e47bfc9976368887ad85bcab920f3d5ca37bf28b4a14b463a61f71f5705607077d8eb7a21d111bff390508367c05caf85bfa6b91fdbb78902b1cc27e24ed274c456f77e8c08fa6086c5473c4097241377d579e0661a0e5d0c220c2f1c133019e9cbab3d49804d0e1e51379926d65dc28a2a77e3227b3c8574978845d68bc8e3a2009c1f78cab1a56a5bdf0e7101c2069e9f450aa2d671a24f8c22a97b7c505e58a8d0f584adca28e082826c6b35b0510966fd7d8d654bf73f55392eb3c29f51a34156f94e63fb563c1707fe4f0ec99e06e2d6e61c7390084aed4334b74a24b690f4b3078bc88ca62c29f10065cf23100c9fc9a74ac389b2ccb112cfd9c7f244e306d7c18ad64112a7fb97721b256fd4a740841a26a4682f34c4b739f734389598bd4c03d910c3e3ecaab5846dc3f7a6cd4c04f8e75ad7a7ebbbbdd7c88856c0d9508c4fd17e4348b8b6d37b3baa849c1bfe14ff7e616d3cf8fb9f25e41a9c364a77ffd47e17eb69285bd1e1d1588e7101a52cc67ff28d49f3f679958f1f6557b02ac52d459d9402ce167ae510f07d6823b12c5ab53ea2ff3e8b808e14e7617886cabd14eefbd5240170acdead50e240a259c026812454c242f63a9705ab4b25edd5f7234ccfdbaacc1bbaee7326666e4829010723c0bbab7ee0b4adf4f063e1b71f531f7847ea1c336d42358b37d398795a0fc5f239aa36131af6488eb6c032f2f2985073a4ec1b2ee7820c53ca42c5bc3dc8fa2fca0b460b4e4ee927885004fd2776670b5ae12c927c6a6b9067132e71eaba5d28981620534c6f97086a747eb8489aa0fb33b42ed74aacc0c01fa5619f48d8cc35af60e4d28608f4de93798905b3755c20931b9b190baeabb2f66d40684124ec44d467b474c55e4eb7cb8d6ffb31fed60f431da8ff665da4f8bb878570964a412d72e74720fa1fbfbfe11dfa6795dadc7db02f9ad62e6f93d3da8c1948f6664cf0f723f944741b83565cc2a78a82631a806026db5c4ea68f322c03b722812aacd56379833a14b288725502603bd540628808f9b6b0d085aeb3190a42a4b8d2d61b7ee4da8d68b5f8cc21d822a3413fa21497f0700b41b98fb6ea0ae570572a4b8a851dc8f8075cc2add19485271278d9f3d8ac4aa04175e65bb3278e33276afe0373f55e8fe67ece00a11de6ce42eafff13b56d14edbff5025c6a170f592e45608c8d0651e5026dda50df3afdf680486e5db5529834bf338d580df00f3f14d5e59565575eeb1aaea2521415a6926fa4963b16488b17767a279d457877cc71c6bbf77a636507baa85954bf889fac3823a3e6350e48d151b30311f5f08376f4d153fbba6dde1556ee9cacd47137f20112196f99e33a2a8fc2334e10b561c02ec70eb863adca5d7a988f33c00fa21473c8c2eecc8332e0b05ac4c6407e5987b94985b72531dd945391e4720fd0b33199e35866c9b68d88664cd81c9073af42c796d29fa8b575379c44336dc12a18d9faf7934d767cd156fed42c204c1f0739fb76fe11ee69a766be3257cd3cefe40af817797662e4424cb9fc7226eb23f14249bde2ae7bff15313d47208d72722b46a7042bddad08522b77c0e04c200cff19dcf78ecf34ce8763ddc95f67c01284887a3b3b45adaebe97040ff871f336e37d86f27be5dca7b8c98e95f0f6eded509a4337ea8a2ef3d8313fe8b413fe06199542f8cac04433f02775cd4df62a412d26613977ad7870daeb8f2fa50f7340dda3fe8490858ae686de7ed19068818b57ca9d31beb413368b0a31a247a5d22775c33e1e684aec3f4b696311f2d1cca69f09ff7bc98c4f2e67184affcf8dfc2e8f55a643ffc2666f13d669dc7f999dd75f722824fdb2cdf388147be3c1b72853160214d1b4eb0fc4c3563df53f6f848161de8643979853c32eaad84e9abaca21d896aacfba9c10f79f709afe4845395a5a6114a9b8424ea67a5de6493c295c1c6df434fae8a017163c7e355d46dd7cfe4a2ae8035900c280c2f347faf62fb503d90673e812081296117f1195e67121c6c03893ed3a784a0a0c2104838c8d179e07df7803528ab6186034c68e24b362b752284f7c22a73d73189af93c4cee07c87ef2bfa84f47d04ab111bb684808722e6aeba4b7cf593af4a0cd05057419d2752618d4f5a0a84b7a75864dbeacb8c78d6c6c94bfebcf29eed3e543c941f1a912616db4551b39a12a84f6f43f89244893a489b15c1af5e3a2f7af01975101fc6321564561d056740d0898a1597e8795fb53eb7cd9f126a992e136ed014ff506287e3d2599cb520204696508bd00c66eb4c70a78e24df1802db88448519eebc4f3d09cadccd4c15b0bad8edfb8181de86bbca1cf0f8b07974676fc5662911398a5168a9eecef932f0eadd9a6bcc26c1ba5640e6a539f6b64692dedc96f44629e74f1ea8bfd23bd66e6666c06b7146ac947d9147b724a1f2c0adee7e86ac1110baaf3c430d1e1d892047c5bebfc3fe47b0bef928141213b943b1ec1ddafaca1b0bd48275e9258ec9449ce4fd1e7d2491ca4a6fb2bae37d6a9e8f67581c6bcbb611a8653383518d565287218c53f250c3e7c40649815f6413c0740b1be6021c4c7b2eaa19a0314355175f39b07c695e83cc1da41cdcbe1e7851db91814f8c3fc2ed9f5c192d696e75109c366e426e4486a707cf27f2cf48fbf1acc38879cc846ae09cf63ebe84698601df4def027b080eedf4f173dbb0282a6028b99042ddd36e27f78fa05bd279ae776f50d83f802bd39e6b4201c02f753186eb105f00fa089d57d25983798610b51358cd9c344dfc57f254bc9af3f34b0ee56913fcfef914f14f97a91387238bd2bd328ff37c2c9d71603fdbde89014a3cb0f2ea39ae86988d6762f99669b70fab81b353a73e6731c60f70f35a8a1da777f760441a1b24981e1cd98ef070d8b4d095f200ecfc3a085f6fa5cf6b2346ed90d8e96e21fa207dbd8facb146e8a3f23964aa6c7c1b7f124ffc2f77306fc6a504bcba9ae2dde7b12ecf77b4370a4afaca183e353b966345d092d254b5ba34e9afc3460fae9f4413bbd58e2ec8518bf243d50f737feec6d167914410218be4e2f9e83de9cf822039bcc4696b282fb4a6d33a6f93570133ca5e1d1f54d14025a1813fd74aa587ba7db0f0974170f28ae378f5d2b2dcaf8d412b75e8310f2c0a92498d4e0211b1327b96cdfd648b3548e342a96cf3770681981d20dd69b2bb404bf615946c0de26cde04666f8ea0b3bbdae5f5f61b25515cb9e7f244496d72536c6db2c7c9db093339418a95deebec3337445391989cf60306101220b4e3ebfafd3a464b577f88b3c700000000000000000000000b12161d262e3840",
		},
	}
	for _, par := range pars {
		pkBytes, _ := hex.DecodeString(par.pk)
		pk, err := par.par.DecodePublicKey(pkBytes)
		if err != nil {
			t.Errorf("DecodePublicKey(%v) failed: %v", par.name, err)
		}
		msgBytes, _ := hex.DecodeString(par.msg)
		ctxBytes, _ := hex.DecodeString(par.ctx)
		sigBytes, _ := hex.DecodeString(par.sig)
		err = pk.Verify(msgBytes, sigBytes, ctxBytes)
		if err != nil {
			t.Errorf("Verify(%v) failed: %v", par.name, err)
		}
	}
}

func TestSignVerify(t *testing.T) {
	pars := []struct {
		name string
		par  *params
	}{
		{"MLDSA44", MLDSA44},
		{"MLDSA65", MLDSA65},
		{"MLDSA87", MLDSA87},
	}
	for _, par := range pars {
		for j := 0; j < numSignVerifyTests; j++ {
			pk, sk := par.par.KeyGen()
			var m [32]byte
			rand.Read(m[:])
			var ctx [32]byte
			rand.Read(ctx[:])
			signature, err := sk.Sign(m[:], ctx[:])
			if err != nil {
				t.Errorf("Sign(%v) failed: %v", par.name, err)
			}
			err = pk.Verify(m[:], signature[:], ctx[:])
			if err != nil {
				t.Errorf("Verify(%v) failed: %v", par.name, err)
			}
		}
	}
}

func TestSignDeterministicVerify(t *testing.T) {
	pars := []struct {
		name string
		par  *params
	}{
		{"MLDSA44", MLDSA44},
		{"MLDSA65", MLDSA65},
		{"MLDSA87", MLDSA87},
	}
	for _, par := range pars {
		for j := 0; j < numSignVerifyTests; j++ {
			pk, sk := par.par.KeyGen()
			var m [32]byte
			rand.Read(m[:])
			var ctx [32]byte
			rand.Read(ctx[:])
			signature, err := sk.SignDeterministic(m[:], ctx[:])
			if err != nil {
				t.Errorf("SignDeterministic(%v) failed: %v", par.name, err)
			}
			err = pk.Verify(m[:], signature[:], ctx[:])
			if err != nil {
				t.Errorf("Verify(%v) failed: %v", par.name, err)
			}
		}
	}
}

func TestSignVerifyWithMu(t *testing.T) {
	pars := []struct {
		name string
		par  *params
	}{
		{"MLDSA44", MLDSA44},
		{"MLDSA65", MLDSA65},
		{"MLDSA87", MLDSA87},
	}
	for _, par := range pars {
		for j := 0; j < numSignVerifyTests; j++ {
			pk, sk := par.par.KeyGen()
			var m [32]byte
			rand.Read(m[:])
			var mu [64]byte
			rand.Read(mu[:])
			signature := sk.SignWithMu(mu)
			err := pk.VerifyWithMu(mu, signature)
			if err != nil {
				t.Errorf("VerifyWithMu(%v) failed", par.name)
			}
		}
	}
}

func TestSignDeterministicVerifyWithMu(t *testing.T) {
	pars := []struct {
		name string
		par  *params
	}{
		{"MLDSA44", MLDSA44},
		{"MLDSA65", MLDSA65},
		{"MLDSA87", MLDSA87},
	}
	for _, par := range pars {
		for j := 0; j < numSignVerifyTests; j++ {
			pk, sk := par.par.KeyGen()
			var m [32]byte
			rand.Read(m[:])
			var mu [64]byte
			rand.Read(mu[:])
			signature := sk.SignDeterministicWithMu(mu)
			err := pk.VerifyWithMu(mu, signature)
			if err != nil {
				t.Errorf("VerifyWithMu(%v) failed", par.name)
			}
		}
	}
}

func TestSignVerifyContextTooLong(t *testing.T) {
	pk, sk := MLDSA44.KeyGen()
	var m [32]byte
	rand.Read(m[:])
	var ctx [256]byte
	rand.Read(ctx[:])
	signature, err := sk.Sign(m[:], ctx[:])
	if err == nil {
		t.Errorf("Sign did not fail with context too long")
	}
	err = pk.Verify(m[:], signature[:], ctx[:])
	if err == nil {
		t.Errorf("Verify did not fail with context too long")
	}
}

func TestSignDeterministicContextTooLong(t *testing.T) {
	_, sk := MLDSA44.KeyGen()
	var m [32]byte
	rand.Read(m[:])
	var ctx [256]byte
	rand.Read(ctx[:])
	_, err := sk.SignDeterministic(m[:], ctx[:])
	if err == nil {
		t.Errorf("Sign did not fail with context too long")
	}
}
