# Copyright 1999-2025 Gentoo Authors
# Distributed under the terms of the GNU General Public License v2

EAPI=8

MY_PN="NetworkManager"
PYTHON_COMPAT=( python3_{11..14} )

inherit linux-info meson-multilib flag-o-matic python-any-r1 \
	readme.gentoo-r1 systemd toolchain-funcs udev vala virtualx

DESCRIPTION="A set of co-operative tools that make networking simple and straightforward"
HOMEPAGE="
	https://www.networkmanager.dev
	https://gitlab.freedesktop.org/NetworkManager/NetworkManager
"
SRC_URI="https://gitlab.freedesktop.org/NetworkManager/NetworkManager/-/releases/${PV}/downloads/${MY_PN}-${PV}.tar.xz"
S="${WORKDIR}"/${MY_PN}-${PV}

LICENSE="GPL-2+ LGPL-2.1+"
SLOT="0"

KEYWORDS="~alpha ~amd64 ~arm ~arm64 ~hppa ~loong ~ppc ~ppc64 ~riscv ~sparc ~x86"

IUSE="audit bluetooth +concheck connection-sharing debug dhclient dhcpcd elogind gnutls gtk-doc +introspection iptables iwd libedit +modemmanager nbft +nss nftables ofono ovs policykit +ppp psl resolvconf selinux syslog systemd teamd test +tools vala +wext +wifi"
RESTRICT="!test? ( test )"

REQUIRED_USE="
	bluetooth? ( modemmanager )
	connection-sharing? ( || ( iptables nftables ) )
	gtk-doc? ( introspection )
	iwd? ( wifi )
	test? ( tools )
	vala? ( introspection )
	wext? ( wifi )
	^^ ( gnutls nss )
	?? ( elogind systemd )
	?? ( dhclient dhcpcd )
	?? ( syslog systemd )
"

COMMON_DEPEND="
	sys-apps/util-linux[${MULTILIB_USEDEP}]
	>=virtual/libudev-175:=[${MULTILIB_USEDEP}]
	sys-apps/dbus[${MULTILIB_USEDEP}]
	net-libs/libndp
	>=dev-libs/glib-2.42:2[${MULTILIB_USEDEP}]
	audit? ( sys-process/audit )
	bluetooth? ( >=net-wireless/bluez-5:= )
	concheck? ( net-misc/curl )
	connection-sharing? (
		net-dns/dnsmasq[dbus,dhcp]
		iptables? ( net-firewall/iptables )
		nftables? ( net-firewall/nftables )
	)
	dhclient? ( >=net-misc/dhcp-4[client] )
	dhcpcd? ( >=net-misc/dhcpcd-9.3.3 )
	elogind? ( >=sys-auth/elogind-219 )
	gnutls? (
		>=net-libs/gnutls-2.12:=[${MULTILIB_USEDEP}]
	)
	introspection? ( >=dev-libs/gobject-introspection-1.82.0-r2:= )
	modemmanager? (
		net-misc/mobile-broadband-provider-info
		>=net-misc/modemmanager-0.7.991:0=
	)
	nbft? ( >=sys-libs/libnvme-1.5 )
	nss? (
		dev-libs/nspr[${MULTILIB_USEDEP}]
		>=dev-libs/nss-3.11[${MULTILIB_USEDEP}]
	)
	ofono? ( net-misc/ofono )
	ovs? ( >=dev-libs/jansson-2.7:= )
	policykit? ( >=sys-auth/polkit-0.106 )
	ppp? ( >=net-dialup/ppp-2.4.5:=[ipv6(+)] )
	psl? ( net-libs/libpsl )
	resolvconf? ( virtual/resolvconf )
	selinux? (
		sec-policy/selinux-networkmanager
		sys-libs/libselinux
	)
	systemd? ( >=sys-apps/systemd-209:0= )
	teamd? (
		>=dev-libs/jansson-2.7:=
		>=net-misc/libteam-1.9
	)
	tools? (
		>=dev-libs/jansson-2.7:=
		>=dev-libs/newt-0.52.15
		libedit? ( dev-libs/libedit )
		!libedit? ( sys-libs/readline:= )
	)
"
RDEPEND="${COMMON_DEPEND}
	acct-group/plugdev
	|| (
		net-misc/iputils[arping(+)]
		net-analyzer/arping
	)
	wifi? (
		!iwd? ( >=net-wireless/wpa_supplicant-0.7.3-r3[dbus] )
		iwd? ( net-wireless/iwd )
	)
"
DEPEND="${COMMON_DEPEND}
	>=sys-kernel/linux-headers-3.18
	net-libs/libndp[${MULTILIB_USEDEP}]
	ppp? ( elibc_musl? ( net-libs/ppp-defs ) )
	test? ( >=dev-libs/jansson-2.7 )
"
BDEPEND="
	dev-util/gdbus-codegen
	dev-util/glib-utils
	>=sys-devel/gettext-0.17
	virtual/pkgconfig
	gtk-doc? (
		dev-util/gtk-doc
		app-text/docbook-xml-dtd:4.1.2
	)
	introspection? (
		$(python_gen_any_dep 'dev-python/pygobject:3[${PYTHON_USEDEP}]')
		dev-lang/perl
		dev-libs/libxslt
	)
	vala? ( $(vala_depend) )
	test? (
		$(python_gen_any_dep '
			dev-python/dbus-python[${PYTHON_USEDEP}]
			dev-python/pygobject:3[${PYTHON_USEDEP}]')
	)
"

python_check_deps() {
	if use introspection; then
		python_has_version "dev-python/pygobject:3[${PYTHON_USEDEP}]" || return
	fi
	if use test; then
		python_has_version "dev-python/dbus-python[${PYTHON_USEDEP}]" &&
		python_has_version "dev-python/pygobject:3[${PYTHON_USEDEP}]"
	fi
}

pkg_setup() {
	if use connection-sharing; then
		if kernel_is lt 5 1; then
			CONFIG_CHECK="~NF_NAT_IPV4 ~NF_NAT_MASQUERADE_IPV4"
		else
			CONFIG_CHECK="~NF_NAT ~NF_NAT_MASQUERADE"
		fi
		linux-info_pkg_setup
	fi

	if use introspection || use test; then
		python-any-r1_pkg_setup
	fi
}

src_prepare() {
	DOC_CONTENTS="To modify system network connections without needing to enter the
		root password, add your user account to the 'plugdev' group."

	default
	use vala && vala_setup

	sed -i \
		-e 's#/usr/bin/sed#/bin/sed#' \
		data/84-nm-drivers.rules \
		|| die
}

meson_nm_program() {
	usex "$1" "-D${2:-$1}=$3" "-D${2:-$1}=no"
}

meson_nm_native_program() {
	multilib_native_usex "$1" "-D${2:-$1}=$3" "-D${2:-$1}=no"
}

multilib_src_configure() {
	# Workaround for LLD on musl systems (bug #959603)
	append-ldflags $(test-flags-CCLD -Wl,--undefined-version)

	# LTO is restricted in older clang for unclear reasons.
	# https://gitlab.freedesktop.org/NetworkManager/NetworkManager/-/issues/593
	# https://gitlab.freedesktop.org/NetworkManager/NetworkManager/-/merge_requests/2053
	tc-is-clang && [[ $(clang-major-version) -lt 18 ]] && filter-lto

	# Follow order of options in meson_options.txt
	local emesonargs=(
		--localstatedir="${EPREFIX}/var" # overrride eclass ${EPREFIX}/var/lib

		# system paths
		-Dsystemdsystemunitdir=$(systemd_get_systemunitdir)
		-Dsystem_ca_path=/etc/ssl/certs
		-Dudev_dir=$(get_udevdir)
		-Ddbus_conf_dir=/usr/share/dbus-1/system.d
		-Dkernel_firmware_dir=/lib/firmware
		-Diptables=/sbin/iptables
		-Dnft=/sbin/nft
		-Ddnsmasq=/usr/sbin/dnsmasq

		# platform
		-Ddist_version=${PVR}
		$(meson_native_use_bool policykit polkit)
		$(meson_native_use_bool policykit config_auth_polkit_default)
		-Dmodify_system=true
		-Dpolkit_agent_helper_1=/usr/lib/polkit-1/polkit-agent-helper-1
		$(meson_native_use_bool selinux)
		$(meson_native_use_bool systemd systemd_journal)
		-Dconfig_wifi_backend_default=$(multilib_native_usex iwd iwd default)
		-Dhostname_persist=gentoo
		-Dlibaudit=$(multilib_native_usex audit)

		# features
		$(meson_native_use_bool wext)
		$(meson_native_use_bool wifi)
		$(meson_native_use_bool iwd)
		$(meson_native_use_bool ppp)
		-Dpppd=/usr/sbin/pppd
		$(meson_native_use_bool modemmanager modem_manager)
		$(meson_native_use_bool ofono)
		$(meson_native_use_bool concheck)
		$(meson_native_use_bool teamd teamdctl)
		$(meson_native_use_bool ovs)
		$(meson_native_use_bool tools nmcli)
		$(meson_native_use_bool tools nmtui)
		$(meson_native_use_bool tools nm_cloud_setup)
		$(meson_native_use_bool bluetooth bluez5_dun)
		# ebpf is problematic in at least v1.46.0, bug #926943
		-Debpf=false
		$(meson_native_use_bool nbft)

		# configuration plugins
		-Dconfig_plugins_default=keyfile
		-Difcfg_rh=false
		-Difupdown=false
		-Dconfig_migrate_ifcfg_rh_default=false

		# handlers for resolv.conf
		$(meson_nm_native_program resolvconf "" /sbin/resolvconf)
		-Dnetconfig=no
		-Dconfig_dns_rc_manager_default=auto

		# dhcp clients
		$(meson_nm_program dhclient "" /sbin/dhclient)
		$(meson_nm_program dhcpcd "" /sbin/dhcpcd)

		# miscellaneous
		$(meson_native_use_bool introspection)
		$(meson_native_use_bool vala vapi)
		$(meson_native_use_bool gtk-doc docs)
		-Dtests=$(multilib_native_usex test)
		$(meson_native_true firewalld_zone)
		-Dmore_asserts=0
		$(meson_use debug more_logging)
		-Dvalgrind=no
		-Dvalgrind_suppressions=
		-Dld_gc=false
		$(meson_native_use_bool psl libpsl)
		-Dqt=false
	)

	if multilib_is_native_abi && use systemd; then
		emesonargs+=( -Dsession_tracking_consolekit=false )
		emesonargs+=( -Dsession_tracking=systemd )
		emesonargs+=( -Dsuspend_resume=systemd )
	elif multilib_is_native_abi && use elogind; then
		emesonargs+=( -Dsession_tracking_consolekit=false )
		emesonargs+=( -Dsession_tracking=elogind )
		emesonargs+=( -Dsuspend_resume=elogind )
	else
		emesonargs+=( -Dsession_tracking_consolekit=false )
		emesonargs+=( -Dsession_tracking=no )
		emesonargs+=( -Dsuspend_resume=auto )
	fi

	if multilib_is_native_abi && use syslog; then
		emesonargs+=( -Dconfig_logging_backend_default=syslog )
	elif multilib_is_native_abi && use systemd; then
		emesonargs+=( -Dconfig_logging_backend_default=journal )
	else
		emesonargs+=( -Dconfig_logging_backend_default=default )
	fi

	if multilib_is_native_abi && use dhclient; then
		emesonargs+=( -Dconfig_dhcp_default=dhclient )
	elif multilib_is_native_abi && use dhcpcd; then
		emesonargs+=( -Dconfig_dhcp_default=dhcpcd )
	else
		emesonargs+=( -Dconfig_dhcp_default=internal )
	fi

	if use nss; then
		emesonargs+=( -Dcrypto=nss )
	else
		emesonargs+=( -Dcrypto=gnutls )
	fi

	if use tools ; then
		emesonargs+=( -Dreadline=$(usex libedit libedit libreadline) )
	else
		emesonargs+=( -Dreadline=none )
	fi

	# Same hack as net-dialup/pptpd to get proper plugin dir for ppp, bug #519986
	if use ppp; then
		local PPPD_VER=`best_version net-dialup/ppp`
		PPPD_VER=${PPPD_VER#*/*-} #reduce it to ${PV}-${PR}
		PPPD_VER=${PPPD_VER%%[_-]*} # main version without beta/pre/patch/revision
		emesonargs+=( -Dpppd_plugin_dir=/usr/$(get_libdir)/pppd/${PPPD_VER} )
	fi

	meson_src_configure
}

multilib_src_test() {
	if use test && multilib_is_native_abi; then
		python_setup
		virtx meson_src_test
	fi
}

multilib_src_install() {
	meson_src_install
	if ! multilib_is_native_abi; then
		rm -r "${ED}"/{etc,usr/{bin,lib/NetworkManager,share},var} || die
	fi
}

multilib_src_install_all() {
	! use systemd && readme.gentoo_create_doc

	newinitd "${FILESDIR}/init.d.NetworkManager-r3" NetworkManager
	newconfd "${FILESDIR}/conf.d.NetworkManager" NetworkManager

	# Need to keep the /etc/NetworkManager/dispatched.d for dispatcher scripts
	keepdir /etc/NetworkManager/dispatcher.d

	# Provide openrc net dependency only when nm is connected
	exeinto /etc/NetworkManager/dispatcher.d
	newexe "${FILESDIR}/10-openrc-status-r4" 10-openrc-status
	sed -e "s:@EPREFIX@:${EPREFIX}:g" \
		-i "${ED}/etc/NetworkManager/dispatcher.d/10-openrc-status" || die

	keepdir /etc/NetworkManager/system-connections
	chmod 0600 "${ED}"/etc/NetworkManager/system-connections/.keep* # bug #383765, upstream bug #754594

	# Allow users in plugdev group to modify system connections
	insinto /usr/share/polkit-1/rules.d/
	doins "${FILESDIR}"/01-org.freedesktop.NetworkManager.settings.modify.system.rules

	insinto /usr/lib/NetworkManager/conf.d #702476
	doins "${S}"/examples/nm-conf.d/31-mac-addr-change.conf

	if use iwd; then
		# This goes to $nmlibdir/conf.d/ and $nmlibdir is '${prefix}'/lib/$PACKAGE, thus always lib, not get_libdir
		cat <<-EOF > "${ED}"/usr/lib/NetworkManager/conf.d/iwd.conf || die
		[device]
		wifi.backend=iwd
		EOF
	fi

	mv "${ED}"/usr/share/doc/{NetworkManager/examples/,${PF}} || die
	rmdir "${ED}"/usr/share/doc/NetworkManager || die

	# Empty
	rmdir "${ED}"/var{/lib{/NetworkManager,},} || die

	# https://gitlab.freedesktop.org/NetworkManager/NetworkManager/-/issues/1653
	# https://gitlab.freedesktop.org/NetworkManager/NetworkManager/-/merge_requests/2068
	# prebuilt manpages aren't installed by meson
	use gtk-doc || doman man/*.[1578]
}

pkg_postinst() {
	udev_reload

	systemd_reenable NetworkManager.service
	! use systemd && readme.gentoo_print_elog

	if [[ -e "${EROOT}/etc/NetworkManager/nm-system-settings.conf" ]]; then
		ewarn "The ${PN} system configuration file has moved to a new location."
		ewarn "You must migrate your settings from ${EROOT}/etc/NetworkManager/nm-system-settings.conf"
		ewarn "to ${EROOT}/etc/NetworkManager/NetworkManager.conf"
		ewarn
		ewarn "After doing so, you can remove ${EROOT}/etc/NetworkManager/nm-system-settings.conf"
	fi

	# NM fallbacks to plugin specified at compile time (upstream bug #738611)
	# but still show a warning to remember people to have cleaner config file
	if [[ -e "${EROOT}/etc/NetworkManager/NetworkManager.conf" ]]; then
		if grep plugins "${EROOT}/etc/NetworkManager/NetworkManager.conf" | grep -q ifnet; then
			ewarn
			ewarn "You seem to use 'ifnet' plugin in ${EROOT}/etc/NetworkManager/NetworkManager.conf"
			ewarn "Since it won't be used, you will need to stop setting ifnet plugin there."
			ewarn
		fi
	fi

	# NM shows lots of errors making nmcli almost unusable, bug #528748 upstream bug #690457
	if grep -r "psk-flags=1" "${EROOT}"/etc/NetworkManager/; then
		ewarn "You have psk-flags=1 setting in above files, you will need to"
		ewarn "either reconfigure affected networks or, at least, set the flag"
		ewarn "value to '0'."
	fi

	if use dhclient || use dhcpcd; then
		ewarn "You have enabled USE=dhclient and/or USE=dhcpcd, but NetworkManager since"
		ewarn "version 1.20 defaults to the internal DHCP client. If the internal client"
		ewarn "works for you, and you're happy with, the alternative USE flags can be"
		ewarn "disabled. If you want to use dhclient or dhcpcd, then you need to tweak"
		ewarn "the main.dhcp configuration option to use one of them instead of internal."
		# https://gitlab.freedesktop.org/NetworkManager/NetworkManager/-/merge_requests/1988
		ewarn
		ewarn "Note that dhclient has been deprecated and support for that will be removed"
		ewarn "in a future release."
	fi
}

pkg_postrm() {
	udev_reload
}
